/*:
 * @target MZ
 * @plugindesc [MZ] ピクチャを左上配置のまま中心基準で拡大・ゆらゆら回転（補正付き）
 * @author you+ChatGPT
 *
 * @help
 * ■概要
 *  - 「ピクチャ表示」を左上(0,0)など通常どおり置いたまま、
 *    見た目は“中心基準”で拡大縮小＆回転させます。
 *  - サイン波の「ゆらゆら回転」と「スケール脈動」を同時適用可能。
 *
 * ■使い方
 *  1) いつも通り「ピクチャ表示」（原点：左上推奨）
 *  2) 本プラグインのコマンド「中心基準にする（左上固定補正あり）」を実行
 *  3) 必要に応じて「ゆらゆら回転 開始」「スケール脈動 開始」
 *  4) 以降は「ピクチャの移動」で拡大率を触れば、見た目が中心から拡縮します
 *
 * ■注意
 *  - 同じく Sprite_Picture の座標やアンカー・スケールを毎フレーム上書きする
 *    プラグインとは競合の可能性があります。
 *
 * @command CenterPivotOn
 * @text 中心基準にする（左上固定補正あり）
 * @arg pictureId
 * @text ピクチャID
 * @type number
 * @min 1
 * @default 1
 *
 * @command CenterPivotOff
 * @text 中心基準を解除（通常に戻す）
 * @arg pictureId
 * @text ピクチャID
 * @type number
 * @min 1
 * @default 1
 *
 * @command WobbleStart
 * @text ゆらゆら回転 開始
 * @arg pictureId
 * @type number
 * @min 1
 * @default 1
 * @arg amplitudeDeg
 * @text 振幅（度）
 * @type number
 * @decimals 2
 * @default 5
 * @arg frequencyHz
 * @text 周波数（Hz）
 * @type number
 * @decimals 3
 * @default 0.7
 * @arg phaseDeg
 * @text 位相（度）
 * @type number
 * @decimals 2
 * @default 0
 *
 * @command WobbleStop
 * @text ゆらゆら回転 停止
 * @arg pictureId
 * @type number
 * @min 1
 * @default 1
 *
 * @command PulseStart
 * @text スケール脈動 開始
 * @arg pictureId
 * @type number
 * @min 1
 * @default 1
 * @arg ampPercentX
 * @text 振幅X（±%）
 * @type number
 * @decimals 2
 * @default 5
 * @arg ampPercentY
 * @text 振幅Y（±%）
 * @type number
 * @decimals 2
 * @default 5
 * @arg frequencyHz
 * @text 周波数（Hz）
 * @type number
 * @decimals 3
 * @default 0.9
 * @arg phaseDeg
 * @text 位相（度）
 * @type number
 * @decimals 2
 * @default 0
 *
 * @command PulseStop
 * @text スケール脈動 停止
 * @arg pictureId
 * @type number
 * @min 1
 * @default 1
 */

(() => {
  const PLUGIN_NAME = "CenterPivotPictureMZ";

  // -----------------------------
  // Helpers
  // -----------------------------
  const toRad = deg => (deg * Math.PI) / 180;
  const TWO_PI = Math.PI * 2;

  // -----------------------------
  // Game_Picture: extra state
  // -----------------------------
  const _Game_Picture_initBasic = Game_Picture.prototype.initBasic;
  Game_Picture.prototype.initBasic = function() {
    _Game_Picture_initBasic.call(this);
    this._centerPivotKeepTL = false; // “左上に置いたまま中心基準で見せる”フラグ

    // 原点記憶（0:左上 / 1:中央）
    this._origOrigin = undefined;

    // ゆらゆら回転
    this._wobbleOn = false;
    this._wobbleAmpDeg = 0;
    this._wobbleHz = 0;
    this._wobblePhase = 0; // rad

    // スケール脈動
    this._pulseOn = false;
    this._pulseAmpX = 0; // percent
    this._pulseAmpY = 0; // percent
    this._pulseHz = 0;
    this._pulsePhase = 0; // rad
  };

  Game_Picture.prototype.setCenterPivotKeepTL = function(on) {
    this._centerPivotKeepTL = !!on;
  };
  Game_Picture.prototype.centerPivotKeepTL = function() {
    return this._centerPivotKeepTL;
  };

  Game_Picture.prototype._rememberOrigin = function() {
    // _origin は 0:左上 / 1:中央（MZ標準）
    this._origOrigin = this._origin;
  };

  Game_Picture.prototype.startWobble = function(ampDeg, hz, phaseDeg) {
    this._wobbleOn = true;
    this._wobbleAmpDeg = Number(ampDeg) || 0;
    this._wobbleHz = Number(hz) || 0;
    this._wobblePhase = toRad(Number(phaseDeg) || 0);
  };
  Game_Picture.prototype.stopWobble = function() {
    this._wobbleOn = false;
  };
  Game_Picture.prototype.wobbleRotationRad = function(t) {
    if (!this._wobbleOn) return 0;
    return toRad(this._wobbleAmpDeg) * Math.sin(TWO_PI * this._wobbleHz * t + this._wobblePhase);
  };

  Game_Picture.prototype.startPulse = function(ampX, ampY, hz, phaseDeg) {
    this._pulseOn = true;
    this._pulseAmpX = Number(ampX) || 0;
    this._pulseAmpY = Number(ampY) || 0;
    this._pulseHz = Number(hz) || 0;
    this._pulsePhase = toRad(Number(phaseDeg) || 0);
  };
  Game_Picture.prototype.stopPulse = function() {
    this._pulseOn = false;
  };
  Game_Picture.prototype.pulseScaleAdd = function(t) {
    if (!this._pulseOn) return { addX: 0, addY: 0 };
    const s = Math.sin(TWO_PI * this._pulseHz * t + this._pulsePhase);
    return {
      addX: this._pulseAmpX * s, // percent
      addY: this._pulseAmpY * s
    };
  };

  // -----------------------------
  // Plugin Commands
  // -----------------------------
  PluginManager.registerCommand(PLUGIN_NAME, "CenterPivotOn", args => {
    const id = Number(args.pictureId || 1);
    const pic = $gameScreen.picture(id);
    if (pic) {
      pic._rememberOrigin();           // 原点を記録
      pic.setCenterPivotKeepTL(true);  // 中心基準(左上固定補正)ON
    }
  });

  PluginManager.registerCommand(PLUGIN_NAME, "CenterPivotOff", args => {
    const id = Number(args.pictureId || 1);
    const pic = $gameScreen.picture(id);
    if (pic) {
      pic.setCenterPivotKeepTL(false); // 通常に戻す
    }
  });

  PluginManager.registerCommand(PLUGIN_NAME, "WobbleStart", args => {
    const id = Number(args.pictureId || 1);
    const pic = $gameScreen.picture(id);
    if (pic) pic.startWobble(args.amplitudeDeg, args.frequencyHz, args.phaseDeg);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "WobbleStop", args => {
    const id = Number(args.pictureId || 1);
    const pic = $gameScreen.picture(id);
    if (pic) pic.stopWobble();
  });

  PluginManager.registerCommand(PLUGIN_NAME, "PulseStart", args => {
    const id = Number(args.pictureId || 1);
    const pic = $gameScreen.picture(id);
    if (pic) pic.startPulse(args.ampPercentX, args.ampPercentY, args.frequencyHz, args.phaseDeg);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "PulseStop", args => {
    const id = Number(args.pictureId || 1);
    const pic = $gameScreen.picture(id);
    if (pic) pic.stopPulse();
  });

  // -----------------------------
  // Sprite_Picture: 既定の上書きを抑止し、最終段で統一制御
  // -----------------------------
  const _SP_updateScale = Sprite_Picture.prototype.updateScale;
  Sprite_Picture.prototype.updateScale = function() {
    const picture = this.picture();
    if (picture && picture.centerPivotKeepTL && picture.centerPivotKeepTL()) {
      // スケールは updateOther で最終決定するため何もしない
      return;
    }
    _SP_updateScale.call(this);
  };

  const _SP_updateOrigin = Sprite_Picture.prototype.updateOrigin;
  Sprite_Picture.prototype.updateOrigin = function() {
    const picture = this.picture();
    if (picture && picture.centerPivotKeepTL && picture.centerPivotKeepTL()) {
      // アンカーも updateOther で決定するため何もしない
      return;
    }
    _SP_updateOrigin.call(this);
  };

  const _SP_updateOther = Sprite_Picture.prototype.updateOther;
  Sprite_Picture.prototype.updateOther = function() {
    _SP_updateOther.call(this);

    const picture = this.picture();
    if (!picture) return;

    const t = Graphics.frameCount / 60;

    // ---- スケール（ベース% + 脈動%）を最終決定 ----
    const baseScaleX = picture.scaleX(); // percent
    const baseScaleY = picture.scaleY(); // percent
    const pulse = picture.pulseScaleAdd(t); // {addX, addY} in percent
    const finalScaleX = (baseScaleX + pulse.addX) / 100;
    const finalScaleY = (baseScaleY + pulse.addY) / 100;
    this.scale.x = finalScaleX;
    this.scale.y = finalScaleY;

    // ---- 回転（ベース角 + ゆらゆら） ----
    const baseRot = (picture.angle() * Math.PI) / 180;
    const wobbleRot = picture.wobbleRotationRad(t);
    this.rotation = baseRot + wobbleRot;

    // ---- アンカー＆位置補正 ----
    if (picture.centerPivotKeepTL && picture.centerPivotKeepTL()) {
      // 見た目の中心基準
      this.anchor.x = 0.5;
      this.anchor.y = 0.5;

      const bmp = this.bitmap;
      if (bmp && bmp.isReady()) {
        const w = bmp.width;
        const h = bmp.height;

        // ピクチャ表示の原点が左上のときだけ「左上固定補正」を適用
        const needTLComp = (picture._origOrigin === 0 || picture._origOrigin === undefined);
        if (needTLComp) {
          // 中心アンカーなので、中心 = 左上 + (サイズ*スケール*0.5)

 this.x = picture.x() + w * 0.5; // 中心を常に同じ座標に固定
 this.y = picture.y() + h * 0.5; // スケールに依存させない

        } else {
          // もともと中央原点なら補正不要
          this.x = picture.x();
          this.y = picture.y();
        }
      } else {
        // 未読込時は一旦そのまま（読込後に自動で補正される）
        this.x = picture.x();
        this.y = picture.y();
      }
    } else {
      // 通常（左上基準）
      this.anchor.x = 0;
      this.anchor.y = 0;
      this.x = picture.x();
      this.y = picture.y();
    }

    // 不透明度/ブレンドは標準に委ねる
    this.opacity = picture.opacity();
    this.blendMode = picture.blendMode();
  };
})();
